<?php
/**
 * Create A Simple Theme Options Panel
 *
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
 exit;
}

// Start Class
if ( ! class_exists( 'Caporil_Theme_Options' ) ) {
    class Caporil_Theme_Options {
        /**
         * Start things up
         *
         * @since 1.0.0
         */
        public function __construct() {
            // We only need to register the admin panel on the back-end
            if ( is_admin() ) {
                add_action( 'admin_menu', array( 'Caporil_Theme_Options', 'add_admin_menu' ) );
                add_action( 'admin_init', array( 'Caporil_Theme_Options', 'register_settings' ) );
            }
        }

        /**
         * Returns all theme options
         *
         * @since 1.0.0
         */
        public static function get_theme_options() {
            return get_option( 'theme_caporil_options' );
        }

        /**
         * Returns single theme option
         *
         * @since 1.0.0
         */
        public static function get_theme_option( $id ) {
            $options = self::get_theme_options();
            if ( isset( $options[$id] ) ) {
                return $options[$id];
            }
        }

        /**
         * Add sub menu page
         *
         * @since 1.0.0
         */
        public static function add_admin_menu() {
            add_menu_page(
                    esc_html__( 'Theme Settings', 'caporil' ),
                    esc_html__( 'Theme Settings', 'caporil' ),
                        'manage_options',
                        'theme-settings',
                        array( 'Caporil_Theme_Options', 'create_admin_page' )
                    );
        }

        /**
         * Register a setting and its sanitization callback.
         *
         * We are only registering 1 setting so we can store all options in a single option as
         * an array. You could, however, register a new setting for each option
         *
         * @since 1.0.0
         */
        public static function register_settings() {
            register_setting( 'theme_caporil_options', 'theme_caporil_options', array( 'Caporil_Theme_Options', 'sanitize' ) );
        }

        /**
         * Sanitization callback
         *
         * @since 1.0.0
         */
        public static function sanitize( $options ) {
            // If we have options lets sanitize them
            if ( $options ) {
                // Static generation
                if ( ! empty( $options['static'] ) ) {
                    $options['static'] = 'on';
                } else {
                    unset( $options['static'] ); // Remove from options if not checked
                }
                // Owner
                if ( ! empty( $options['owner'] ) ) {
                    $options['owner'] = sanitize_text_field( $options['owner'] );
                } else {
                    $options['owner'] = '';
                }
                // Suppress version tag line
                if ( ! empty( $options['suppressVersionTag'] ) ) {
                    $options['suppressVersionTag'] = 'on';
                } else {
                    unset( $options['suppressVersionTag'] );
                }
                // New model, replacement text
                if ( ! empty( $options['newmodel'] ) && ! empty( $options['newname'] )) {
                    $model = sanitize_text_field( $options['newmodel'] );
                    $name = sanitize_text_field( $options['newname'] );
                    global $wpdb;
                    $camera_table = $wpdb->prefix . 'caporil_cameras';
                    $wpdb->replace($camera_table, array( 'model' => $model, 'name' => $name ) );
                }
                unset ( $options['newmodel'] );
                unset ( $options['newname'] );
            }
            // Return sanitized options
            return $options;
        }

        /**
         * Settings page output
         *
         * @since 1.0.0
         */
        public static function create_admin_page() { ?>
            <div class="wrap">
                <h1><?php esc_html_e( 'Theme Options', 'caporil' ); ?></h1>
                <form method="post" action="options.php">
                    <?php settings_fields( 'theme_caporil_options' ); ?>
                    <table class="form-table wpex-custom-admin-login-table">
                        <tr valign="top">
                            <th scope="row"><?php esc_html_e( 'Static Web Site', 'caporil' ); ?></th>
                            <td>
                                <?php $value = self::get_theme_option( 'static' ); ?>
                                <input type="checkbox" name="theme_caporil_options[static]" <?php checked( $value, 'on' ); ?>> <?php esc_html_e( 'Tick if this theme is used on a site that will be statically generated', 'caporil' ); ?>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><?php esc_html_e( 'Copyright Owner', 'caporil' ); ?></th>
                            <td>
                                <?php $value = self::get_theme_option( 'owner' ); ?>
                                <input type="text" name="theme_caporil_options[owner]" value="<?php echo $value?>"> <?php esc_html_e( 'The copyright owner displayed in the footer of every page', 'caporil' ); ?>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><?php esc_html_e( 'Suppress Theme Version Display', 'caporil' ); ?></th>
                            <td>
                                <?php $value = self::get_theme_option( 'suppressVersionTag' ); ?>
                                <input type="checkbox" name="theme_caporil_options[suppressVersionTag]" <?php checked( $value, 'on' ); ?>> <?php esc_html_e( 'Tick if the theme tag in the footer should not include the version number', 'caporil' ); ?>
                            </td>
                        </tr>
                    </table>
                    <p>Many cameras provide only a cryptic model name as the camera name in the image metadata. You can provide replacement text for these model names using the table below:
                    <table class="wp-list-table widefat fixed striped table-view-list">
                        <thead><tr><th>Cryptic Model Name</th><th>Replacement text</th></tr></thead>
                        <tbody><tr><td><input type="text" name="theme_caporil_options[newmodel]" value="" placeholder="New cryptic model name"></td>
                        <td><input type="text" name="theme_caporil_options[newname]" value="" placeholder="Replacement text for the model name"></td></tr>
                        <?php global $wpdb;
                            $camera_table = $wpdb->prefix . 'caporil_cameras';
                            $cameras = $wpdb->get_results( "SELECT model,name FROM $camera_table ORDER BY id DESC" );
                            foreach ( $cameras as $camera ) {
                                echo "<tr><td>" . $camera->model . "</td><td>" . $camera->name . "</td></tr>";
                            }
                        ?></tbody>
                    </table>
                    <?php submit_button(); ?>
                </form>
            </div><!-- .wrap -->
        <?php }
    }
}
new Caporil_Theme_Options();
// Helper function to use in your theme to return a theme option value
function caporil_get_theme_option( $id = '' ) {
    return Caporil_Theme_Options::get_theme_option( $id );
}
