/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals Util, ShipDesignType, RepairShipOrder, ScrapShipOrder, ShipOrder, StandingOrders, GameMap */
/* exported VesselOrderMaker */
class VesselOrderMaker {
	constructor() {
		const self = this;
		self._modal = document.getElementById("vessel-orders");
		self._name1 = document.getElementById("vessel-orders-vessel-name1");
		self._name2 = document.getElementById("vessel-orders-vessel-name2");
		self._name3 = document.getElementById("vessel-orders-vessel-name3");
		self._rank = document.getElementById("vessel-orders-captain-rank");
		self._startLocation = document.getElementById("vessel-orders-start-location");
		self._canJump = document.getElementById("vessel-orders-jump-capable");
		self._retreating = document.getElementById("vessel-orders-retreating");
		self._retreatingLocation = document.getElementById("vessel-orders-retreat-location");
		self._notRetreating = document.getElementById("vessel-orders-not-retreating");
		self._jumpOrHold = document.getElementById("vessel-orders-jump-or-hold");
		self._holdPositionButton = document.getElementById("vessel-orders-hold-position");
		self._changeJumpButton = document.getElementById("vessel-orders-change-jump");
		self._jumpInstructions = document.getElementById("vessel-orders-jumpInstructions");
		self._cancelJumpButton = document.getElementById("vessel-orders-cancelJumpChange");
		self._orderedToJump = document.getElementById("vessel-orders-jumping");
		self._action = document.getElementById("vessel-orders-action");
		self._doNothingButton = document.getElementById("vessel-orders-do-nothing");
		self._repairButton = document.getElementById("vessel-orders-repair");
		self._repairCost = document.getElementById("vessel-orders-repair-cost");
		self._scrapButton = document.getElementById("vessel-orders-scrap");
		self._scrapValue = document.getElementById("vessel-orders-scrap-value");
		self._coloniseButton = document.getElementById("vessel-orders-colonise");
		self._embarkButton = document.getElementById("vessel-orders-embark");
		self._disembarkButton = document.getElementById("vessel-orders-disembark");
		self._planetBustButton = document.getElementById("vessel-orders-planet-bust");
		self._dest1 = document.getElementById("vessel-orders-destination1");
		self._dest2 = document.getElementById("vessel-orders-destination2");
		self._dest3 = document.getElementById("vessel-orders-destination3");
		self._dest4 = document.getElementById("vessel-orders-destination4");
		self._standing = document.getElementById("vessel-orders-standing-orders");
		self._standingCantRetreat =  document.getElementById("vessel-orders-cant-retreat");
		self._standingAlwaysRetreat =  document.getElementById("vessel-orders-standing-retreat-always");
		self._standingBounceRetreat =  document.getElementById("vessel-orders-standing-retreat-bounce");
		self._standingEngage =  document.getElementById("vessel-orders-standing-engage");
		self._standingEngageBounce =  document.getElementById("vessel-orders-standing-engage-bounce");
		self._standingRemainWhileCovered =  document.getElementById("vessel-orders-standing-retreat-if-no-cover");
		self._standingRemainWhileCoveredBounce =  document.getElementById("vessel-orders-standing-bounce-if-no-cover");
		self._hasWeapons = document.getElementById("vessel-orders-has-weapons");
		self._and = document.getElementById("vessel-orders-and");
		self._canRetreat = document.getElementById("vessel-orders-can-retreat");
		self._sittingDuck = document.getElementById("vessel-orders-sitting-duck");
		self._confirm = document.getElementById("vessel-orders-confirm");
		self._cancel = document.getElementById("vessel-orders-cancel");
		self._oldOrders = document.getElementById("vessel-orders-old-orders");

		self._holdPositionButton.onclick = function() {
			self._ship.order.cancelJump();
			self._finishJump();
		};
		self._changeJumpButton.onclick = function() {
			self._jumpInstructions.classList.remove("hidden");
			const location = self._ship.location;
			const jumpRange = self._ship.getJumpRange();
			for(let row=1; row<21; row++) {
				for(let column of "ABCDEFGHIJKLMNOP") {
					const name = column+row;
					const hexelement = document.getElementById(name);
					if (GameMap.range(location,name) <= jumpRange && self._player.map.hex(name).onGrid) {
						hexelement.onclick = function() { self._selectJumpTarget(name); };
					} else {
						hexelement.classList.add("greyout");
					}
				}
			}
			self._modal.onclose = self._close.bind(self);
		};
		self._cancelJumpButton.onclick = function() {
			self._finishJump();
		};
		self._doNothingButton.onclick = function() {
			if (self._ship.order instanceof RepairShipOrder || self._ship.order instanceof ScrapShipOrder) {
				self._ship.order = new ShipOrder();
				self._changeJumpButton.disabled = false;
			}
			self._ship.order.colonise = false;
			self._ship.order.resetMarines();
			self._ship.order.bust = false;
			self._fillInFields();
		};
		self._repairButton.onclick = function() {
			self._ship.order = new RepairShipOrder(self._ship.getRepairCost());
			self._changeJumpButton.disabled = true;
			self._finishJump();
		};
		self._scrapButton.onclick = function() {
			self._ship.order = new ScrapShipOrder(self._ship.getScrapValue());
			self._changeJumpButton.disabled = true;
			self._finishJump();
		};
		self._coloniseButton.onclick = function() {
			if (self._ship.order instanceof RepairShipOrder || self._ship.order instanceof ScrapShipOrder) {
				self._ship.order = new ShipOrder();
				self._changeJumpButton.disabled = false;
			}
			self._ship.order.colonise = true;
			self._ship.order.resetMarines();
			self._ship.order.bust = false;
			self._fillInFields();
		};
		self._embarkButton.onclick = function() {
			if (self._ship.order instanceof RepairShipOrder || self._ship.order instanceof ScrapShipOrder) {
				self._ship.order = new ShipOrder();
				self._changeJumpButton.disabled = false;
			}
			self._ship.order.colonise = false;
			self._ship.order.setMarinesToEmbark();
			self._ship.order.bust = false;
			self._fillInFields();
		};
		self._disembarkButton.onclick = function() {
			if (self._ship.order instanceof RepairShipOrder || self._ship.order instanceof ScrapShipOrder) {
				self._ship.order = new ShipOrder();
				self._changeJumpButton.disabled = false;
			}
			self._ship.order.colonise = false;
			self._ship.order.setMarinesToDisembark();
			self._ship.order.bust = false;
			self._fillInFields();
		};
		self._planetBustButton.onclick = function() {
			if (self._ship.order instanceof RepairShipOrder || self._ship.order instanceof ScrapShipOrder) {
				self._ship.order = new ShipOrder();
				self._changeJumpButton.disabled = false;
			}
			self._ship.order.colonise = false;
			self._ship.order.resetMarines();
			self._ship.order.bust = true;
			self._fillInFields();
		};
		self._standing.onchange = function() {
			if (self._standing.selectedIndex == self._standingEngage.index) self._ship.standingOrders = StandingOrders.Engage;
			if (self._standing.selectedIndex == self._standingEngageBounce.index) self._ship.standingOrders = StandingOrders.EngageBounce;
			if (self._standing.selectedIndex == self._standingCantRetreat.index) self._ship.standingOrders = StandingOrders.Remain;
			if (self._standing.selectedIndex == self._standingAlwaysRetreat.index) self._ship.standingOrders = StandingOrders.Retreat;
			if (self._standing.selectedIndex == self._standingBounceRetreat.index) self._ship.standingOrders = StandingOrders.Bounce;
			if (self._standing.selectedIndex == self._standingRemainWhileCovered.index) self._ship.standingOrders = StandingOrders.Escorted;
			if (self._standing.selectedIndex == self._standingRemainWhileCoveredBounce.index) self._ship.standingOrders = StandingOrders.EscortedBounce;
			self._fillInFields();
		};
		self._confirm.onclick = function() {
			self._modal.close();
		};
		self._cancel.onclick = function() {
			self._ship.order = ShipOrder.fromJSON(JSON.parse(self._oldOrder));
			self._callback(); // Get the field in the table updated too
			self._modal.close();
		};
	}
	open(player,ship,callback) {
		// Note: caller should have created the ship order object by this point and initialised it.
		this._player = player;
		this._ship = ship;
		this._oldOrder = JSON.stringify(ship.order);
		this._oldOrderText = ship.order.toString();
		this._callback = callback;
		if (ship.standingOrders == StandingOrders.Engage) this._standing.selectedIndex = this._standingEngage.index;
		if (ship.standingOrders == StandingOrders.EngageBounce) this._standing.selectedIndex = this._standingEngageBounce.index;
		if (ship.standingOrders == StandingOrders.Remain) this._standing.selectedIndex = this._standingCantRetreat.index;
		if (ship.standingOrders == StandingOrders.Retreat) this._standing.selectedIndex = this._standingAlwaysRetreat.index;
		if (ship.standingOrders == StandingOrders.Bounce) this._standing.selectedIndex = this._standingBounceRetreat.index;
		if (ship.standingOrders == StandingOrders.EscortedBounce) this._standing.selectedIndex = this._standingRemainWhileCoveredBounce.index;
		const canChangeJump = this._fillInFields();
		this._modal.open();
		if (canChangeJump) this._changeJumpButton.onclick(); // Do a click on the assumption that the player will want to change location (they can simply either not click or cancel the jump if they don't  want the jump)
	}
	_fillInFields() {
		let canChangeJump = true;
		Util.replaceText(this._name1,this._ship.name); Util.replaceText(this._name2,this._ship.name); Util.replaceText(this._name3,this._ship.name);
		Util.replaceText(this._rank, (this._ship.design.type == ShipDesignType.Ship ? "Captain" : "Commander"));
		Util.replaceText(this._startLocation,this._ship.location);

		this._canJump.classList.toggle("hidden",this._ship.getJumpRange()==0);
		if (this._ship.getJumpRange()==0) { canChangeJump = false; }

		this._retreating.classList.toggle("hidden",!this._ship.retreatingTo);
		Util.replaceText(this._retreatingLocation,this._ship.retreatingTo ? this._ship.retreatingTo : "");
		this._notRetreating.classList.toggle("hidden",this._ship.retreatingTo);
		if (this._ship.retreatingTo) { canChangeJump = false; }

		if (this._ship.order.jump) {
			Util.replaceText(this._jumpOrHold,`jump to ${this._ship.order.jump}`);
		} else {
			Util.replaceText(this._jumpOrHold,`hold position at ${this._ship.location}`);
		}
		this._orderedToJump.classList.toggle("hidden",!this._ship.order.jump);
		this._fillInAction();

		const hex = this._player.map.hex(this._ship.location);
		this._repairButton.disabled = true; // Initial assumption
		if (this._ship.retreatingTo) {
			this._repairButton.title = "Note: A ship cannot be repaired while it is retreating";
		} else if (!hex.planet && this._ship.type == ShipDesignType.Ship) {
			this._repairButton.title = "Note: A ship cannot be repaired in empty space (it must be at one of your class 20 or better colonies)";
		} else if (hex.owner !== this._player.stylename) {
			this._repairButton.title = "Note: A ship cannot be repaired at an enemy world (it must be at one of your class 20 or better colonies)";
		} else if (this._ship.design.type == ShipDesignType.Ship && hex.level < 20) {
			this._repairButton.title = `Note: A ship cannot be repaired at this class ${hex.level} colony (it must be class 20 or better)`;
		} else if (this._ship._order.jump) {
			this._repairButton.title = "Note: A ship cannot be repaired in the same turn as it is jumping";
		} else {
			this._repairButton.disabled = false;
			this._repairButton.title = "";
		}

		if (this._ship.retreatingTo) {
			this._scrapButton.disabled = true;
			this._scrapButton.title = "Note: A ship cannot be scrapped in the same turn as it is retreating";
		} else  if (this._ship.order.jump) {
			this._scrapButton.disabled = true;
			this._scrapButton.title = "Note: A ship cannot be scrapped in the same turn as it is jumping";
		} else  {
			this._scrapButton.disabled = false;
		}

		this._repairButton.classList.toggle("hidden",this._ship.getRepairCost()==0);
		this._coloniseButton.classList.toggle("hidden",this._ship.getColonyImprovement()==0);
		this._embarkButton.classList.toggle("hidden",this._ship.getMaxMarines()==this._ship.marines);
		this._disembarkButton.classList.toggle("hidden",this._ship.marines==0);
		this._planetBustButton.classList.toggle("hidden",!this._ship.canPlanetBust());
		Util.replaceText(this._repairCost,this._ship.getRepairCost());
		Util.replaceText(this._scrapValue,this._ship.getScrapValue());
		const dest = (this._ship.order.jump ? this._ship.order.jump : this._ship.location);
		Util.replaceText(this._dest1,dest); Util.replaceText(this._dest2,dest); Util.replaceText(this._dest3,dest); Util.replaceText(this._dest4,dest);

		this._hasWeapons.classList.toggle("hidden",!this._ship.hasWeapons(true,true));
		this._and.classList.toggle("hidden",!this._ship.hasWeapons(true,true) || this._ship.getJumpRange() == 0);
		this._canRetreat.classList.toggle("hidden",this._ship.getJumpRange()==0);
		this._sittingDuck.classList.toggle("hidden",this._ship.hasWeapons(true,true) || this._ship.getJumpRange() > 0);
		Util.replaceText(this._oldOrders,this._oldOrderText);

		const selected = this._standing.selectedOptions.item(0);
		if (selected == this._standingAlwaysRetreat) { Util.replaceText("vessel-orders-standing-orders-extra"," as soon as possible"); }
		if (selected == this._standingBounceRetreat) { Util.replaceText("vessel-orders-standing-orders-extra"," as soon as possible, returning to the battle on the next turn"); }
		if (selected == this._standingCantRetreat) { Util.replaceText("vessel-orders-standing-orders-extra"," until either you or they are destroyed"); }
		if (selected == this._standingEngage) { Util.replaceText("vessel-orders-standing-orders-extra"," the enemy while you are capable of attack. If you are unable to attack you are to retreat as soon as possible"); }
		if (selected == this._standingEngageBounce) { Util.replaceText("vessel-orders-standing-orders-extra"," the enemy while you are capable of attack. If you are unable to attack you are to escape into jump space as soon as possible, returning to the battle on the next turn"); }
		if (selected == this._standingRemainWhileCovered) { Util.replaceText("vessel-orders-standing-orders-extra"," while escorted by other vessels that are capable of attacking the enemy. If no friendly vessels are capable of attack you are to retreat as soon as possible"); }
		if (selected == this._standingRemainWhileCoveredBounce) { Util.replaceText("vessel-orders-standing-orders-extra"," while escorted by other vessels that are capable of attacking the enemy. If no friendly vessels are capable of attack you are to escape into jump space as soon as possible, returning to the battle on the next turn"); }
		this._callback(); // And get the field in the table updated too

		return canChangeJump;
	}
	_fillInAction() {
		let action = "do nothing";
		if (this._ship.order instanceof RepairShipOrder) {
			action = "assist the repair crews as they repair your vessel";
		} else if (this._ship.order instanceof ScrapShipOrder) {
			action = "support the deconstruction crew as they scrap your vessel";
		} else if (this._ship.order.colonise) {
			action = "colonise (or join) any planet in your hex";
		} else if (this._ship.order.marinesEmbarking) {
			action = "embark as many marines as possible";
		} else if (this._ship.order.marinesDisembarking) {
			action = "disembark your marines if there is a colony present in your hex";
		}
		Util.replaceText(this._action,action);
	}
	_selectJumpTarget(target) {
		if (target !== this._ship.location) {
			this._ship.order.jump = target;
		} else {
			this._ship.order.cancelJump();
		}
		this._finishJump();
	}
	_finishJump() {
		this._jumpInstructions.classList.add("hidden");
		for(let row=1; row<21; row++) {
			for(let column of "ABCDEFGHIJKLMNOP") {
				const name = column+row;
				const hexelement = document.getElementById(name);
				hexelement.onclick = null;
				hexelement.classList.remove("greyout");
			}
		}
		const canChangeJump = this._fillInFields();
		if (canChangeJump) this._changeJumpButton.disabled = false;
		this._modal.onclose = null;
	}
	_close() {
		if (this._modal.onclose) {
			this._finishJump();
		}
	}

}