/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals Util, TECHS, RangedCombo, ShipDesigner, ShipDesignType, PlayerResearch, BuildVessel */
/* globals CarryForwardOrder, BuyVehicleOrder, BuyStarBaseOrder, AddMarinesOrder, ImproveColonyOrder */
/* exported LinkedRange, PlayerOrders */

class LinkedRange {
	_getNode(id) {
		return (typeof id === "string" ? document.getElementById(id) : id);
	}
	constructor(sliderElementId,inputElementId,costRowId,costElementId,cancelElementId,updateFunction) {
		const self = this;
		const row = this._getNode(costRowId);
		const cost = this._getNode(costElementId);
		const cancel = this._getNode(cancelElementId);

		self._combo = new RangedCombo(sliderElementId,inputElementId);
		self._multiplier = 1;

		self._combo.oninput = () => {
			row.dataset.cost = Math.ceil(self._combo.value * self._multiplier);
			Util.replaceText(cost,row.dataset.cost);
			if (updateFunction) {
				updateFunction(self._combo.value);
			}
			PlayerOrders.recomputeTotals();
		};
		cancel.onclick = () => {
			self._combo.value = 0;
		};
	}
	get multiplier() { return this._multiplier; }
	set multiplier(m) { this._multiplier = m; }
	get max() { return this._combo.max; }
	set max(m) { this._combo.max = m; }
	get value() { return this._combo.value; }
	set value(v) { this._combo.value = v; }
	reset() {
		this._combo.value = 0;
	}
}

class CostRow {
	constructor(purchase,cost,cancelFunction) {
		const self = this;
		const table = document.getElementById("purchase-table");
		this._row = table.insertRow();
		this._purchaseCell = document.createElement("td"); this._purchaseCell.append(purchase);
		this._costCell = document.createElement("td"); this._costCell.append(cost);
		this._cancelCell = document.createElement("td");
		const cancel = document.createElement("button"); cancel.append("Cancel"); this._cancelCell.append(cancel);
		cancel.onclick = function() {
			cancelFunction(table,self._row.rowIndex,self);
			PlayerOrders.recomputeTotals();
		};

		this._row.dataset.cost = cost;
		this._row.append(this._purchaseCell,this._costCell,this._cancelCell);
		PlayerOrders.recomputeTotals();
	}
	get row() { return this._row; }
	get purchaseCell() { return this._purchaseCell; }
	updatePurchase(newText) {
		Util.replaceText(this._purchaseCell,newText);
	}
	get costCell() { return this._costCell; }
	get cancelCell() { return this._cancelCell; }
	setCost(cost) {
		this._row.dataset.cost = cost;
		Util.replaceText(this._costCell,cost);
		PlayerOrders.recomputeTotals();
	}
}

var PlayerOrders = {
	_player: null,
	_leftOver: 0,
	_vehiclePurchaseOrders: [],
	_additionalMarines: {},
	_improvedColonies: [],
	_shipModal: null,
	_spaceBaseModal: null,
	_starBaseModal: null,
	buyVessel: function(design,shipname,location,first,cost) {
		const order = (design.type == ShipDesignType.StarBase ? new BuyStarBaseOrder(design,shipname,location) : new BuyVehicleOrder(design,shipname,location) );
		if (first) order.setFirstOfDesign();
		PlayerOrders._vehiclePurchaseOrders.push(order);
		new CostRow(`Buy a new ${design.name} to be called '${shipname}' at ${location}`,
			cost,
			function(table,rowIndex,_costRow) {
				PlayerOrders._vehiclePurchaseOrders = PlayerOrders._vehiclePurchaseOrders.filter(o => o !== order);
				table.deleteRow(rowIndex);
			}
		);
	},
	init: function() {
		PlayerResearch.init();
		PlayerOrders._shipModal = new BuildVessel("Ship",ShipDesignType.Ship,false,PlayerOrders.buyVessel);
		PlayerOrders._spaceBaseModal = new BuildVessel("SpaceBase",ShipDesignType.SpaceBase,false,PlayerOrders.buyVessel);
		PlayerOrders._starBaseModal = new BuildVessel("StarBase",ShipDesignType.StarBase,true,PlayerOrders.buyVessel);

		const newstarbase = document.getElementById("buildStarBase");
		newstarbase.onclick = function() {
			PlayerOrders._starBaseModal.open(PlayerOrders._player.capital);
		};

	},
	setPlayer: function(player) {
		// Save the reference to the player
		PlayerOrders._player = player;
		ShipDesigner.setPlayer(player);
		PlayerResearch.setPlayer(player);
		PlayerOrders._shipModal.setPlayer(player);
		PlayerOrders._spaceBaseModal.setPlayer(player);
		PlayerOrders._starBaseModal.setPlayer(player);
		PlayerOrders.recomputeTotals();
	},
	setStandardShipDesigns: function(list) {
		PlayerOrders._shipModal.setStandardShipDesigns(list);
		PlayerOrders._spaceBaseModal.setStandardShipDesigns(list);
		PlayerOrders._starBaseModal.setStandardShipDesigns(list);
	},
	prepare: function(baseMap) {
		// Clear out the purchase table
		const table = document.getElementById("purchase-table");
		let templaterow = table.querySelector(".template.last");
		while (templaterow.nextSibling) {
			templaterow.parentNode.removeChild(templaterow.nextSibling);
		}
		// Production and reset research expenditure to zero
		PlayerOrders._leftOver = PlayerOrders._player.production+PlayerOrders._player.carriedProduction;
		Util.replaceText("savedPPs",PlayerOrders._player.carriedProduction);
		Util.replaceText("productionThisTurn",PlayerOrders._player.production);
		Util.replaceText("availablePPs",PlayerOrders._leftOver);

		// Update researchs
		PlayerResearch.prepare(PlayerOrders._leftOver);

		// Update the colony reports (and hide fields as necessary)
		PlayerOrders._additionalMarines = {};
		PlayerOrders._improvedColonies = [];
		PlayerOrders._prepareColonies(baseMap);

		document.getElementById("tooMuchExpenditure").classList.add("hidden");
		document.getElementById("nextTurn").disabled = false;
		PlayerOrders._vehiclePurchaseOrders = [];
		PlayerOrders.recomputeTotals();
	},
	_prepareColonies: function(baseMap) {
		// Find the colony listing table
		let colonies = document.getElementById("colonies");

		// Clear existing rows after the templates
		let templaterow = colonies.querySelector(".template.last");
		while (templaterow.nextSibling) {
			templaterow.parentNode.removeChild(templaterow.nextSibling);
		}
		let templates = colonies.querySelectorAll(".template");

		let capitalHex = baseMap.hex(PlayerOrders._player.capital);
		let convertId = function(element,hexid) {
			if (element && element.id) { element.id = element.id.replace(/@@/g,hexid); }
			for(let child of element.childNodes) {
				convertId(child,hexid);
			}
		};

		PlayerOrders._player.map.forEachHex(function(hex) {
			if (hex.owner == PlayerOrders._player.stylename && hex.level > 0) {
				for(let template of templates) {
					let newrow = template.cloneNode(true);
					newrow.classList.remove("template","last");
					convertId(newrow,hex.id);
					colonies.appendChild(newrow);
				}
				if (hex.id == PlayerOrders._player.capital) {
					Util.replaceText("colony-"+hex.id+"-name",hex.id+" (capital)");
				} else {
					Util.replaceText("colony-"+hex.id+"-name",hex.id);
				}
				const report = document.createElement("p");
				report.append(hex.colonyReport ? hex.colonyReport : "Nothing to report");
				if (hex.interdicted || capitalHex.interdicted) {
					const bold = document.createElement("b");
					if (capitalHex.interdicted && hex.id != capitalHex.id) {
						bold.append("Capital is interdicted - All production here is lost");
					} else if (hex.id != capitalHex.id) {
						bold.append("Colony is interdicted - All production here is lost");
					}
					report.append(document.createElement("br"));
					report.append(bold);
				}
				document.getElementById("colony-"+hex.id+"-report").appendChild(report);
				Util.replaceText("colony-"+hex.id+"-marines",hex.marines);
				Util.replaceText("colony-"+hex.id+"-class",hex.level);
				Util.replaceText("colony-"+hex.id+"-production",hex.level*20);

				const colony = hex.id;
				// Purchase Ships -----------------------------------------------------------------------------
				const shipButton = document.getElementById("colony-"+hex.id+"-build");
				if (hex.level < 20) {
					shipButton.disabled = true;
					shipButton.title = "A colony must be at least class 20 to be able to build ships";
				} else {
					shipButton.onclick = function() {
						PlayerOrders._shipModal.open(colony);
					};
				}
				// Purchase Marines -----------------------------------------------------------------------------
				const improveButton = document.getElementById("colony-"+hex.id+"-improve");
				PlayerOrders._additionalMarines[colony] = 0;
				const marineCostRow= new CostRow(`Purchase 0 Marine companies at ${colony}`,0,function(_table,_rowIndex,_costRow) {
					PlayerOrders._additionalMarines[colony] = 0;
				});
				// sliderElementId,inputElementId,costRowId,costElementId,cancelElementId
				const linkedRange = new LinkedRange("colony-"+hex.id+"-marines-range",
								"colony-"+hex.id+"-marines-number",
								marineCostRow.row,
								marineCostRow.costCell,
								marineCostRow.cancelCell,
								function(companies) {
									marineCostRow.updatePurchase(`Purchase ${companies} Marine compan${companies == 1 ? "y" : "ies"} at ${colony}`);
									PlayerOrders._additionalMarines[colony] = companies;
								});
				linkedRange.multiplier = 10;
				linkedRange.max = Math.floor(PlayerOrders._leftOver/10);

				// Improve Colony  -----------------------------------------------------------------------------
				if (hex.level < 5) {
					improveButton.disabled = true;
					improveButton.title = "A colony must be at least class 5 to be improved by spending production points";
				} else {
					const improveRow = new CostRow(`Improve colony at ${colony}`,0,function(_table,_rowIndex,costRow) {
						improveButton.innerHTML = "Improve Colony Class";
						PlayerOrders._improvedColonies = PlayerOrders._improvedColonies.filter(id => id !== colony);
						costRow.setCost(0);
					});
					improveButton.onclick = function() {
						if (PlayerOrders._improvedColonies.includes(colony)) {
							improveButton.innerHTML = "Improve Colony Class";
							PlayerOrders._improvedColonies = PlayerOrders._improvedColonies.filter(id => id !== colony);
							improveRow.setCost(0);
						} else {
							improveButton.innerHTML = "Cancel Colony Improvement";
							PlayerOrders._improvedColonies.push(colony);
							improveRow.setCost(hex.level*20);
						}
					};
				}
				// Purchase SpaceBases  -----------------------------------------------------------------------------
				const baseButton = document.getElementById("colony-"+hex.id+"-buildbase");
				if (PlayerOrders._player.researchLevel(TECHS.planetaryDefences) < 3) {
					baseButton.disabled = true;
					baseButton.classList.add("hidden");
				} else {
					baseButton.onclick = function() {
						PlayerOrders._spaceBaseModal.open(colony);
					};
				}
			}
		});
	},
	getNumberOfVesselsOrdered(design) {
		let number = 0;
		for(let order of PlayerOrders._vehiclePurchaseOrders) {
			if (((order instanceof BuyVehicleOrder) || (order instanceof BuyStarBaseOrder)) && (order.design === design)) {
				number++;
			}
		}
		return number;
	},
	isNameOnAnOrder: function(proposedName) {
		let duplicate = false;
		for(let order of PlayerOrders._vehiclePurchaseOrders) {
			if (((order instanceof BuyVehicleOrder) || (order instanceof BuyStarBaseOrder)) && (order.name === proposedName)) {
				duplicate = true; break;
			}
		}
		return duplicate;
	},
	isAVesselOnOrder: function() {
		for(let order of PlayerOrders._vehiclePurchaseOrders) {
			if (order instanceof BuyVehicleOrder) return true;
		}
		return false;
	},
	getOrders: function() {
		let orders = PlayerOrders._vehiclePurchaseOrders.concat(PlayerResearch.getOrders()); // concat creates a new array
		for(let colonyBuyingMarines of Object.keys(PlayerOrders._additionalMarines)) {
			if (PlayerOrders._additionalMarines[colonyBuyingMarines] > 0) {
				orders.push(new AddMarinesOrder(colonyBuyingMarines,PlayerOrders._additionalMarines[colonyBuyingMarines]));
			}
		}
		for(let colony of PlayerOrders._improvedColonies) {
			orders.push(new ImproveColonyOrder(colony));
		}
		orders.push(new CarryForwardOrder(PlayerOrders._leftOver));
		return orders;
	},
	recomputeTotals: function() {
		if (PlayerOrders._player) {
			const table = document.getElementById("purchase-table");
			let total = 0;
			for(let row of table.rows) {
				if (row.dataset && row.dataset.cost !== undefined) {
					total += parseInt(row.dataset.cost);
				}
			}
			Util.replaceText("totalExpenditure",total);
			if (total > PlayerOrders._player.production+PlayerOrders._player.carriedProduction) {
				document.getElementById("tooMuchExpenditure").classList.remove("hidden");
				document.getElementById("carriedForwardRow").classList.add("red");
				document.getElementById("nextTurn").disabled = true;
			} else {
				document.getElementById("tooMuchExpenditure").classList.add("hidden");
				document.getElementById("carriedForwardRow").classList.remove("red");
				document.getElementById("nextTurn").disabled = false;
			}

			document.getElementById("noExpenditure").classList.toggle("hidden",total!==0);

			PlayerOrders._leftOver = PlayerOrders._player.production+PlayerOrders._player.carriedProduction-total;
			Util.replaceText("carriedForwardPPs",PlayerOrders._leftOver);
		} /* else invoked before the player was set (possibly because the browser was overeager to fire an oninput event??) */
	},
};
